/************************************************************************
 *
 * \file: aautoDemo.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Android Auto - Prototype
 *
 * \author: D. Girnus / ADIT/SW1/Brunel / dgirnus@de.adit-jv.com
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <string>
#include <aauto/AautoLogging.h>
#include <adit_dlt.h>
#include <adit_macros.h>
#include <sys/prctl.h>
#include "utils/AAutoLayerManager.h"
#include "utils/SignalHandler.h"
#include "Server.h"
#include "ServerMonitor.h"
#include "AutoSmoketest.h"

#include "Testbed.h"

#ifndef DLT_TEST_AAUTO_DEMO
#define DLT_TEST_AAUTO_DEMO "TAAD"
#endif

using namespace adit::aauto;

LOG_DECLARE_CONTEXT(demo);

typedef struct inputArg_t
{
    std::string inLocation;
    std::string inCfgPath;
}inputArg;

void* mainAautoThread(void* argInput)
{
    inputArg* input = (inputArg*) argInput;
    int ret = 1;

    LOG_INFO((demo, "enter main thread"));

    if (!Server::instance().init(input->inCfgPath))
    {
        LOG_FATAL((demo, "failed to start AndroidAuto server"));
        ret = 1;
        pthread_exit(&ret); // auto clean-up
    }

    // the monitor is sending switching and session requests to the server
    ServerMonitor monitor;

    // register SIGINT so that monitor and server can be stopped
    SignalHandler::instance().registerForSIGINT(&Server::instance());
    SignalHandler::instance().registerForSIGINT(&monitor);
    SignalHandler::instance().registerForSIGINT(&AutoSmoketest::instance());

    // set test location to use with sensor source test
    monitor.setLocation(input->inLocation);

    if (!monitor.start())
    {
        LOG_FATAL((demo, "failed to start device monitor"));
        ret = 1;
        pthread_exit(&ret); // auto clean-up
    }

    prctl(PR_SET_NAME, "aauto-demo-main", 0, 0, 0);

    if (AutoSmoketest::instance().getTestMode() == AUTOMATIC)
    {
        AutoSmoketest::instance().waitForExit();

        /* stop timer */
        AutoSmoketest::instance().deleteSignalTimer();

        if (AutoSmoketest::instance().getTestError() == DONE)
        {
            LOG_INFO((demo, "%s()  Wait for %d seconds to get graphic dump",
                    __func__, AutoSmoketest::instance().getDumpDelay()));
            sleep(AutoSmoketest::instance().getDumpDelay());

            /* Request to stop server */
            Server::instance().requestStop();

            /* Request to stop monitor */
            monitor.requestStop();
        }
    }
    else
    {
        monitor.waitForExit();
    }
    LOG_INFO((demo, "stop main loop"));

    if (AutoSmoketest::instance().getTestMode() == AUTOMATIC)
    {
        LOG_INFO((demo,"AST: reset all connected phones "));
        auto& usbDevList = AutoSmoketest::instance().getUsbDevInfoList();
        for(auto& devInfo : usbDevList)
        {
            adit::uspi::DeviceInfo ddInfo = devInfo->getInfo();
            LOG_INFO((demo,"reset device %s", to_string(ddInfo).c_str() ));
            monitor.resetDevice(devInfo, true);
        }
    }
    // stop the monitor
    monitor.stop();

    // signal handler MUST be unregistered before they go out of scope
    SignalHandler::instance().unregister(&AutoSmoketest::instance());
    SignalHandler::instance().unregister(&monitor);
    SignalHandler::instance().unregister(&Server::instance());

    // make sure that everything is cleaned before library clean-up starts

    LOG_INFO((demo, "clean up main loop"));
    Server::instance().cleanUp();

    // clean-up protobuf
    google::protobuf::ShutdownProtobufLibrary();

    if(Testbed::instance().getTestbedMode())
    {
        Testbed::instance().adit_testbed_deinit();
    }

    if (AutoSmoketest::instance().getTestMode() == AUTOMATIC)
    {
        if (AutoSmoketest::instance().getTestError() == DONE)
        {
            ret = 0;
            //for Console
            fprintf (stderr,"\x1B[32m" "\n %s \n" "\x1B[0m", AutoSmoketest::instance().getErrorString().c_str());
            fprintf(stderr, "\x1B[32m" " AAUTO SMOKETEST PASSED !!! \n" "\x1B[0m");
        }
        else
        {
            ret = 1;
            //for console
            fprintf(stderr,"\x1B[31m" "\n %s \n" "\x1B[0m", AutoSmoketest::instance().getErrorString().c_str());
            fprintf(stderr,"\x1B[31m" " AAUTO SMOKETEST FAILED !!! \n" "\x1B[0m");
        }

        AutoSmoketest::instance().setTestResult(ret);
    }

    LOG_INFO((demo, "exit main thread"));
    return nullptr;
}

int main(int argc, char** argv)
{
    inputArg input;
    input.inLocation = "";
    input.inCfgPath = "";

    int autoTestTime = 0;
    int delay = 2;
    int ret = 0;
    int opt_c;

    pthread_t aautoThread;

    AutoSmoketest::instance().setTestMode(MANUAL); //default
  
    while((opt_c = getopt(argc, argv, "hl:f:s:d:g:t")) != -1)
    {
        switch(opt_c)
        {
            case 'h':
                fprintf(stderr, "Usage: aauto-demo [-l <city name>] [-h] \n");
                fprintf(stderr, " -l <city name>   : set city name to make navigation test.(e.g.Hildesheim)\n");
                fprintf(stderr, " -h               : describe option for this command.\n");
                fprintf(stderr, " -f <config path> : path to PFCFG configuration file. (e.g. /etc/pfcfg/aauto-demo.cfg)\n");
                fprintf(stderr, " -s <timeout>     : Automatic smoketest mode with timeout \n");
                fprintf(stderr, " -d <display ID>  : Display ID or Screen ID \n");
                fprintf(stderr, " -t               : Testbed mode.\n");
                fprintf(stderr, " -g <dump delay>  : AST Graphic Dump Delay \n");
                return 0;

            case 's':
                AutoSmoketest::instance().setTestMode(AUTOMATIC); //automated smoketest mode.
                AutoSmoketest::instance().setTestError(NODEVICE); //starting
                autoTestTime = atoi(optarg);
                AutoSmoketest::instance().setSmoketestTimeout(autoTestTime);

                fprintf(stderr, "\x1B[32m" "\n AAUTO demo running as automated smoketest (AST).\n" "\x1B[0m");
                fprintf(stderr, "\x1B[32m" " AAUTO automated smoketest uses only one connected device for testing.\n\n" "\x1B[0m");

                AutoSmoketest::instance().createSignalTimer(); //timerid is now valid
                AutoSmoketest::instance().startSignalTimer();

                break;

            case 'd':
                AutoSmoketest::instance().setDisplayId(atoi(optarg));
                break;

            case 'l':
                input.inLocation = optarg;
                fprintf(stderr, "Set location data to %s \n", input.inLocation.c_str());
                break;

            case 'f':
                input.inCfgPath = optarg;
                fprintf(stderr, "Use config file %s \n", input.inCfgPath.c_str());
                break;

            case 'g':
                delay = atoi(optarg);
                fprintf(stderr, "aauto demo graphic dump delay set to %d Seconds \n", delay);
                AutoSmoketest::instance().setDumpDelay(delay);
                break;

            case 't':
                Testbed::instance().setTestbedMode(TRUE);
                break;

            case ':':
                fprintf(stderr, "Option %c requires an operand\n", optopt);
                break;

            default:
                fprintf(stderr, "Unknown argument %c\n", optopt);
                break;
        }
    }

    // instanciate SignalHandler to set the signal mask
    SignalHandler::instance();

    // register at logging
    aautoRegisterAppWithDLT(DLT_TEST_AAUTO_DEMO, "Android Auto demo");
    LOG_REGISTER_CONTEXT(demo, "ADEM", "Android Auto demo");

    // Create thread for aauto
    pthread_create(&aautoThread, NULL, mainAautoThread, (void*)&input);

    if(Testbed::instance().getTestbedMode())
    {
        Testbed::instance().adit_testbed_init();
        Testbed::instance().adit_testbed_run();
    }

    pthread_join(aautoThread, NULL);
    LOG_INFO((demo, "main thread joined. exit main()"));

    // unregister all AAUTO DLT context of aauto/platform
    LOG_UNREGISTER_CONTEXT(demo);
    // TODO different function names
    aautoDeRegisterAppWithDLT();

    if (AutoSmoketest::instance().getTestMode() == AUTOMATIC)
    {
        ret = AutoSmoketest::instance().getTestResult();
    }
    return ret;
}

